import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.material.DropdownMenuItem
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ArrowDropDown
import androidx.compose.material.icons.filled.ArrowDropUp
import androidx.compose.material3.*
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.layout.onGloballyPositioned
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.toSize
import androidx.compose.ui.window.Window
import androidx.compose.ui.window.WindowState
import androidx.compose.ui.window.application
import com.treemap.AbstractTreeMap
import com.treemap.DefaultTreeMap
import com.treemap.RenderingFactory
import org.mkui.labeling.EnhancedLabel
import org.mkui.palette.PaletteFactory
import org.molap.dataframe.DataFrame
import org.molap.dataframe.JsonDataFrame

internal class Demo

@OptIn(ExperimentalMaterial3Api::class)
fun main() = application {
    Window(onCloseRequest = ::exitApplication, title = "TreeMap for Compose for Desktop", state = WindowState(width = 1000.dp, height = 720.dp)) {
        val inputStream = Demo::class.java.getResourceAsStream("Forbes Global 2000 - 2021.json")
        val dataFrame: DataFrame<Int, String, Any?> = JsonDataFrame.fromInputStream(inputStream)
        val treeMap: AbstractTreeMap<Int, String> = DefaultTreeMap(dataFrame)
        val model = treeMap.model
        val settings = model!!.settings

        // General
        settings.rendering = RenderingFactory.FLAT

        // Group by
        settings.groupByColumns = listOf("Sector", "Industry")

        // Size
        settings.sizeColumn = "Market Value"

        // Color
        settings.colorColumn = "Profits"
        val profitsSettings = settings.getColumnSettings("Profits")
        val negpos = PaletteFactory.instance["negpos"]!!.getPalette()
        val colorMap = model.getColorMap("Profits")
        colorMap!!.palette = negpos
        colorMap.interval!!.setValue(-63.93, 127.86)

        // Label
        val companySettings = settings.getColumnSettings("Company")
//        companySettings.setLabelingFont(CPFont(Font("Helvetica", Font.PLAIN, 9)).nativeFont) // 9 points is the minimum size that will be displayed
        companySettings.setLabelingMinimumCharactersToDisplay(5)
        companySettings.setLabelingResizeTextToFitShape(true)
        companySettings.setLabelingVerticalAlignment(EnhancedLabel.CENTER)
        companySettings.setLabelingHorizontalAlignment(EnhancedLabel.CENTER)

        treeMap.view!!.isShowTiming = true

        var expanded = remember { mutableStateOf(false) }

        MaterialTheme {
            Surface {
                Row {
                    Column(Modifier.fillMaxSize(0.2f).wrapContentSize(Alignment.TopStart)) {
                        var textfieldSize = remember { mutableStateOf(Size.Zero) }

                        val icon = if (expanded.value)
                            Icons.Filled.ArrowDropUp //it requires androidx.compose.material:material-icons-extended
                        else
                            Icons.Filled.ArrowDropDown

                        OutlinedTextField(
                            value = model.settings.sizeColumn?.toString()!!,
                            onValueChange = { model.settings.sizeColumn = it },
                            modifier = Modifier
                                .fillMaxWidth()
                                .onGloballyPositioned { coordinates ->
                                    //This value is used to assign to the DropDown the same width
                                    textfieldSize.value = coordinates.size.toSize()
                                },
                            label = { Text("Size") },
                            trailingIcon = {
                                Icon(icon, "contentDescription",
                                    Modifier.clickable { expanded.value = !expanded.value })
                            }
                        )
                        DropdownMenu(
                            expanded = expanded.value,
                            onDismissRequest = { expanded.value = false },
                            modifier = Modifier
                                .width(with(LocalDensity.current){textfieldSize.value.width.toDp()})
                        ) {
                            model.sizeColumns.forEach { label ->
                                DropdownMenuItem(onClick = {
                                    model.settings.sizeColumn = label.toString()
                                }) {
                                    Text(text = label.toString())
                                }
                            }
                        }
                    }
                    Column(Modifier.fillMaxSize(1f), Arrangement.spacedBy(5.dp)) {
                        treeMap.view!!.component.nativeComponent.invoke()
                    }
                }
            }
        }
    }
}
